<?php

namespace App\Http\Controllers\Admin;

use App\Constants\Status;
use App\Models\Category;
use App\Models\SubCategory;
use Illuminate\Http\Request;
use App\Rules\FileTypeValidate;
use App\Http\Controllers\Controller;

class CategoryController extends Controller
{
    public function mainCategories()
    {
        $pageTitle = 'Categories';
        $categories = Category::searchable(['name'])->latest()->paginate(getPaginate());
        return view('admin.category.categories', compact('pageTitle', 'categories'));
    }

    public function mainCategoriesStore(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'image' => ['image', 'required', new FileTypeValidate(['jpg', 'png', 'jpeg'])]
        ]);

        $category = new Category();
        $category->name = $request->name;
        $category->slug = slug($request->name);
        if ($request->hasFile('image'))
        {
            try
            {
                $category->image = fileUploader($request->image, getFilePath('category'), getFileSize('category'));
            }
            catch (\Exception $exp)
            {
                $notify[] = ['error', 'Couldn\'t upload your image'];
                return back()->withNotify($notify);
            }
        }
        $category->save();
        $notify[] = ['success', 'Category added successfully'];
        return back()->withNotify($notify);
    }

    public function mainCategoriesUpdate(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'image' => ['image', 'nullable', new FileTypeValidate(['jpg', 'png', 'jpeg'])]
        ]);

        $category = Category::findOrFail($request->id);

        $category->name = $request->name;
        $category->slug = slug($request->name);
        if ($request->hasFile('image'))
        {
            try
            {
                $category->image = fileUploader($request->image, getFilePath('category'), getFileSize('category'), $category->image);
            }
            catch (\Exception $exp)
            {
                $notify[] = ['error', 'Couldn\'t upload your image'];
                return back()->withNotify($notify);
            }
        }
        $category->save();
        $notify[] = ['success', 'Category updated successfully'];
        return back()->withNotify($notify);
    }

    public function mainCategoriesStatus($id)
    {
        return Category::changeStatus($id);
    }





    public function subcategories()
    {
        $pageTitle = 'All Sub Categories';
        $subcategories = SubCategory::searchable(['name'])->latest()->whereHas('category')->with('category')->paginate(getPaginate());
        $categories = Category::where('status', Status::ENABLE)->get(['id', 'name']);

        return view('admin.category.subcategories', compact('pageTitle',  'subcategories', 'categories'));
    }

    public function subCategoriesStore(Request $request)
    {
        $request->validate([
            'name'        => 'required',
            'category_id' => 'required|integer|exists:categories,id'
        ]);

        $subcategory              = new SubCategory();
        $subcategory->category_id = $request->category_id;
        $subcategory->name        = $request->name;
        $subcategory->slug        = slug($request->name);
        $subcategory->save();

        $notify[] = ['success', 'Sub Category Added successfully'];
        return back()->withNotify($notify);
    }

    public function subCategoriesUpdate(Request $request)
    {
        $request->validate([
            'name'        => 'required',
            'update_category_id' => 'required|integer|exists:categories,id'
        ]);

        $subcategory              = SubCategory::findOrFail($request->id);
        $subcategory->category_id = $request->update_category_id;
        $subcategory->name        = $request->name;
        $subcategory->slug        = slug($request->name);
        $subcategory->save();

        $notify[] = ['success', 'Sub Category Updated successfully'];
        return back()->withNotify($notify);
    }

    public function subCategoriesStatus($id)
    {
        return SubCategory::changeStatus($id);
    }
}
