<?php

namespace App\Http\Controllers\Admin;

use App\Models\Level;
use App\Models\Course;
use App\Models\Lecture;
use App\Constants\Status;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use App\Http\Controllers\Controller;

class CourseController extends Controller
{
    public function index()
    {
        $pageTitle = "Course Levels";
        $levels = Level::searchable(['name'])->with('courses')->paginate(getPaginate());
        return view('admin.level.list', compact('pageTitle', 'levels'));
    }

    public function levelStore(Request $request)
    {
        $request->validate(['name' => 'required|unique:levels']);

        $level = new Level();
        $level->name = $request->name;
        $level->slug = slug($request->name);
        $level->save();

        $notify[] = ['success', 'New Course Level Added'];
        return back()->withNotify($notify);
    }

    public function levelUpdate(Request $request, $id)
    {
        $request->validate(['name' => 'required|unique:levels,name,' . $id]);

        $level = Level::findOrFail($id);
        $level->name = $request->name;
        $level->update();

        $notify[] = ['success', 'Course Level Updated'];
        return back()->withNotify($notify);
    }

    public function activeCourses()
    {
        $pageTitle = "Active Courses";
        $courses = Course::active()->with(['category', 'subcategory', 'author', 'level'])->searchable(['title', 'code'])->latest()->paginate(getPaginate());

        return view('admin.courses.list', compact('pageTitle', 'courses'));
    }

    public function pendingCourses()
    {
        $pageTitle = "Pending Courses";
        $courses = Course::inactive()->with(['category', 'subcategory', 'author', 'level'])->searchable(['title', 'code'])->latest()->paginate(getPaginate());

        return view('admin.courses.list', compact('pageTitle', 'courses'));
    }

    public function bannedCourses()
    {
        $pageTitle = "Banned Courses";
        $courses = Course::banned()->with(['category', 'subcategory', 'author', 'level'])->searchable(['title', 'code'])->latest()->paginate(getPaginate());

        return view('admin.courses.list', compact('pageTitle', 'courses'));
    }

    public function topCourse($id)
    {
        $course = Course::findOrFail($id);

        if ($course->is_top == Status::YES) {
            $course->is_top =  Status::NO;
            $msg =  'This course has been removed from top course';
        } else {
            $course->is_top = Status::YES;
            $msg =  'This course has been marked as top course';
        }

        $course->save();

        $notify[] = ['success', $msg];
        return back()->withNotify($notify);
    }

    public function courseDetail($id)
    {
        $pageTitle = "Details of this course";
        $course = Course::findOrFail($id);
        return view('admin.courses.course_play', compact('pageTitle', 'course'));
    }



    public function banCourse(Request $request, $id)
    {
        $course = Course::findOrFail($id);

        if ($course->status != 2) {
            $request->validate([
                'reasons' => 'required'
            ]);
            $course->status = 2;
            $course->reasons = $request->reasons;
            $msg =  'Course has been banned';
            $course->save();

            notify($course->author, 'BAN_COURSE', [
                'course_name' => $course->title,
                'reasons' => $request->reasons,
                'time' => showDateTime(Carbon::now(), 'Y-m-d @ h:i A'),
            ]);
        } else {
            $course->status = 1;
            $msg =  'Course has been Un-banned';
            $course->save();

            notify($course->author, 'UN_BAN_COURSE', [
                'course_name' => $course->title,
                'time' => showDateTime(Carbon::now(), 'Y-m-d @ h:i A'),
            ]);
        }

        $notify[] = ['success', $msg];
        return back()->withNotify($notify);
    }



    public function action(Request $request, $id)
    {
        $course = Course::findOrFail($id);

        if ($course->status == 1) {
            $request->validate([
                'reasons' => 'required'
            ]);

            $course->status = 0;
            $course->reasons = $request->reasons;
            $msg =  'Course has been rejected';
            $course->save();

            notify($course->author, 'REJECT_COURSE', [
                'course_name' => $course->title,
                'reasons' => $request->reasons,
                'time' => showDateTime(Carbon::now(), 'Y-m-d @ h:i A'),
            ]);
        } elseif ($course->status == 0) {
            $course->status = 1;
            $msg =  'Course has been approved';
            $course->save();

            notify($course->author, 'APPROVE_COURSE', [
                'course_name' => $course->title,
                'time' => showDateTime(Carbon::now(), 'Y-m-d @ h:i A'),
            ]);
        }

        $notify[] = ['success', $msg];
        return back()->withNotify($notify);
    }

    public function fileDownload($id)
    {
        try {
            $lecture   = Lecture::findOrFail($id);
            $filepath  = getFilePath('lecture') . '/' . $lecture->file;
            $extension = pathinfo($filepath, PATHINFO_EXTENSION);
            $fileName  =  $lecture->slug . '.' . $extension;
            $headers   = [
                'Content-Type'  => 'application',
                'Cache-Control' => 'no-store, no-cache'
            ];
            return response()->download($filepath, $fileName, $headers);
        } catch (\Throwable $th) {
            $notify[] = ['error', 'File Not found'];
            return back()->withNotify($notify);
        }
    }
}
