<?php

namespace App\Http\Controllers\Admin;

use App\Models\User;
use App\Models\Deposit;
use App\Constants\Status;
use App\Models\Withdrawal;
use App\Models\Transaction;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;

class ManageInstructorController extends Controller
{
    public function activeInstructor()
    {
        $pageTitle = 'Manage Active Instructors';
        $users = User::searchable(['username', 'email'])->where('is_instructor', Status::APPROVED)->where('status', Status::ENABLE)->orderBy('id', 'desc')->paginate(getPaginate());
        return view('admin.users.ins_list', compact('pageTitle',  'users'));
    }

    public function pendingInstructor()
    {
        $pageTitle = 'Manage Pending Instructors';
        $users = User::where('is_instructor', Status::PENDING)->where('status', Status::ENABLE)->orderBy('id', 'desc')->paginate(getPaginate());
        return view('admin.users.pending_instructor', compact('pageTitle',  'users'));
    }

    public function detail($id)
    {
        $user = User::findOrFail($id);
        $pageTitle = 'Instructor Detail - ' . $user->username;

        $totalDeposit = Deposit::where('user_id', $user->id)->successful()->sum('amount');
        $totalWithdrawals = Withdrawal::where('user_id', $user->id)->approved()->sum('amount');
        $totalTransaction = Transaction::where('user_id', $user->id)->count();
        $purchasedCourses = $user->userCourses()->count();

        $countries = json_decode(file_get_contents(resource_path('views/partials/country.json')));
        return view('admin.users.detail', compact('pageTitle', 'user', 'totalDeposit', 'totalWithdrawals', 'totalTransaction', 'countries'));
    }

    public function bannedInstructor()
    {
        $pageTitle = 'Banned Instructors';
        $users = User::searchable(['username', 'email'])->where('is_instructor', Status::APPROVED)->where('status', Status::DISABLE)->orderBy('id', 'desc')->paginate(getPaginate());
        return view('admin.users.ins_list', compact('pageTitle', 'users'));
    }

    public function emailUnverifiedInstructor()
    {
        $pageTitle = 'Email Unverified Instructors';
        $users = User::searchable(['username', 'email'])->where('is_instructor', Status::APPROVED)->where('ev', Status::DISABLE)->orderBy('id', 'desc')->paginate(getPaginate());
        return view('admin.users.ins_list', compact('pageTitle',  'users'));
    }

    public function mobileUnverifiedInstructor()
    {
        $pageTitle = 'SMS Unverified Instructors';
        $users = User::searchable(['username', 'email'])->where('is_instructor', Status::APPROVED)->where('sv', Status::DISABLE)->orderBy('id', 'desc')->paginate(getPaginate());
        return view('admin.users.ins_list', compact('pageTitle',  'users'));
    }

    public function kycUnverifiedInstructor()
    {
        $pageTitle = 'KYC Unverified Instructors';
        $users = User::searchable(['username', 'email'])->where('is_instructor', Status::APPROVED)->where('kv', Status::KYC_UNVERIFIED)->orderBy('id', 'desc')->paginate(getPaginate());
        return view('admin.users.list', compact('pageTitle', 'users'));
    }

    public function kycPendingInstructor()
    {
        $pageTitle = 'KYC Pending Instructors';
        $users = User::searchable(['username', 'email'])->where('is_instructor', Status::APPROVED)->where('kv', Status::KYC_PENDING)->orderBy('id', 'desc')->paginate(getPaginate());
        return view('admin.users.list', compact('pageTitle', 'users'));
    }

    public function instructorWithBalance()
    {
        $pageTitle = 'Instructor with balance';
        $users = User::searchable(['username', 'email'])->where('is_instructor', Status::APPROVED)->where('balance', '!=', 0)->orderBy('id', 'desc')->paginate(getPaginate());
        return view('admin.users.ins_list', compact('pageTitle',  'users'));
    }

    public function allInstructor()
    {
        $pageTitle = 'All Instructor';
        $users = User::searchable(['username', 'email'])->where('is_instructor', Status::APPROVED)->orderBy('id', 'desc')->paginate(getPaginate());
        return view('admin.users.ins_list', compact('pageTitle',  'users'));
    }

    public function downloadResume($id)
    {
        $instructor = User::findOrFail($id);

        $filepath = getFilePath('resume') . '/' .  $instructor->instructor_info->resume;

        $extension = pathinfo($filepath, PATHINFO_EXTENSION);
        $fullname = @$instructor->instructor_info->firstname . '_' . $instructor->instructor_info->lastname;
        $fileName =   $fullname . '.' . $extension;
        $headers = [
            'Content-Type' => 'application/octet-stream',
            'Cache-Control' => 'no-store, no-cache'
        ];

        return response()->download($filepath, $fileName, $headers);
    }


    public function instructorApprove($id)
    {
        $instructor = User::findOrFail($id);
        $instructor->is_instructor = Status::APPROVED;
        $instructor->update();
        $notify[] = ['success', 'Request has been approved'];
        return back()->withNotify($notify);
    }

    public function instructorReject($id)
    {
        $instructor = User::findOrFail($id);
        $instructor->is_instructor = Status::REJECTED;
        $instructor->update();
        $notify[] = ['success', 'Request has been rejected'];
        return back()->withNotify($notify);
    }


    public function kycDetails($id)
    {
        $pageTitle = 'KYC Details';
        $user = User::findOrFail($id);
        return view('admin.users.kyc_detail', compact('pageTitle', 'user'));
    }

    public function kycApprove($id)
    {
        $user = User::findOrFail($id);
        $user->kv = Status::KYC_VERIFIED;
        $user->save();

        notify($user, 'KYC_APPROVE', []);

        $notify[] = ['success', 'KYC approved successfully'];
        return to_route('admin.instructors.kyc.pending')->withNotify($notify);
    }

    public function kycReject(Request $request, $id)
    {
        $request->validate([
            'reason' => 'required'
        ]);
        $user = User::findOrFail($id);
        $user->kv = Status::KYC_UNVERIFIED;
        $user->kyc_rejection_reason = $request->reason;
        $user->save();

        notify($user, 'KYC_REJECT', [
            'reason' => $request->reason
        ]);

        $notify[] = ['success', 'KYC rejected successfully'];
        return to_route('admin.instructors.kyc.pending')->withNotify($notify);
    }
}
