<?php

namespace App\Http\Controllers\User;

use App\Models\Form;
use App\Models\Coupon;
use App\Models\Course;
use App\Models\Comment;
use App\Constants\Status;
use App\Lib\FormProcessor;
use App\Models\CouponUser;
use App\Models\DeviceToken;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Validation\Rule;
use App\Lib\GoogleAuthenticator;
use App\Models\AdminNotification;
use App\Http\Controllers\Controller;
use App\Rules\FileTypeValidate;
use Illuminate\Support\Facades\Validator;

class UserController extends Controller
{
    public function home()
    {
        $pageTitle = 'Dashboard';
        $user = auth()->user();
        $trxs = Transaction::where('user_id', auth()->id())->latest()->take(10)->get();
        return view('Template::user.dashboard', compact('pageTitle', 'user', 'trxs'));
    }

    public function depositHistory(Request $request)
    {
        $pageTitle = 'Deposit History';
        $deposits = auth()->user()->deposits()->searchable(['trx'])->with(['gateway'])->orderBy('id', 'desc')->paginate(getPaginate());
        return view('Template::user.deposit_history', compact('pageTitle', 'deposits'));
    }

    public function show2faForm()
    {
        $ga = new GoogleAuthenticator();
        $user = auth()->user();
        $secret = $ga->createSecret();
        $qrCodeUrl = $ga->getQRCodeGoogleUrl($user->username . '@' . gs('site_name'), $secret);
        $pageTitle = '2FA Security';
        return view('Template::user.twofactor', compact('pageTitle', 'secret', 'qrCodeUrl'));
    }

    public function create2fa(Request $request)
    {
        $user = auth()->user();
        $request->validate([
            'key' => 'required',
            'code' => 'required',
        ]);
        $response = verifyG2fa($user, $request->code, $request->key);
        if ($response) {
            $user->tsc = $request->key;
            $user->ts = Status::ENABLE;
            $user->save();
            $notify[] = ['success', 'Two factor authenticator activated successfully'];
            return back()->withNotify($notify);
        } else {
            $notify[] = ['error', 'Wrong verification code'];
            return back()->withNotify($notify);
        }
    }

    public function disable2fa(Request $request)
    {
        $request->validate([
            'code' => 'required',
        ]);

        $user = auth()->user();
        $response = verifyG2fa($user, $request->code);
        if ($response) {
            $user->tsc = null;
            $user->ts = Status::DISABLE;
            $user->save();
            $notify[] = ['success', 'Two factor authenticator deactivated successfully'];
        } else {
            $notify[] = ['error', 'Wrong verification code'];
        }
        return back()->withNotify($notify);
    }

    public function transactions()
    {
        $pageTitle = 'Transactions';
        $remarks = Transaction::distinct('remark')->orderBy('remark')->get('remark');

        $transactions = Transaction::where('user_id', auth()->id())->searchable(['trx'])->filter(['trx_type', 'remark'])->orderBy('id', 'desc')->paginate(getPaginate());

        return view('Template::user.transactions', compact('pageTitle', 'transactions', 'remarks'));
    }

    public function kycForm()
    {
        if (auth()->user()->kv == Status::KYC_PENDING) {
            $notify[] = ['error', 'Your KYC is under review'];
            return to_route('user.home')->withNotify($notify);
        }
        if (auth()->user()->kv == Status::KYC_VERIFIED) {
            $notify[] = ['error', 'You are already KYC verified'];
            return to_route('user.home')->withNotify($notify);
        }
        $pageTitle = 'KYC Form';
        $form = Form::where('act', 'kyc')->first();
        return view('Template::user.kyc.form', compact('pageTitle', 'form'));
    }

    public function kycData()
    {
        $user = auth()->user();
        $pageTitle = 'KYC Data';
        abort_if($user->kv == Status::VERIFIED, 403);
        return view('Template::user.kyc.info', compact('pageTitle', 'user'));
    }

    public function kycSubmit(Request $request)
    {
        $form = Form::where('act', 'kyc')->firstOrFail();
        $formData = $form->form_data;
        $formProcessor = new FormProcessor();
        $validationRule = $formProcessor->valueValidation($formData);
        $request->validate($validationRule);
        $user = auth()->user();
        foreach (@$user->kyc_data ?? [] as $kycData) {
            if ($kycData->type == 'file') {
                fileManager()->removeFile(getFilePath('verify') . '/' . $kycData->value);
            }
        }
        $userData = $formProcessor->processFormData($request, $formData);
        $user->kyc_data = $userData;
        $user->kyc_rejection_reason = null;
        $user->kv = Status::KYC_PENDING;
        $user->save();

        $notify[] = ['success', 'KYC data submitted successfully'];
        return to_route('user.home')->withNotify($notify);
    }

    public function userData()
    {
        $user = auth()->user();

        if ($user->profile_complete == Status::YES) {
            return to_route('user.home');
        }

        $pageTitle  = 'User Data';
        $info       = json_decode(json_encode(getIpInfo()), true);
        $mobileCode = @implode(',', $info['code']);
        $countries  = json_decode(file_get_contents(resource_path('views/partials/country.json')));

        return view('Template::user.user_data', compact('pageTitle', 'user', 'countries', 'mobileCode'));
    }

    public function userDataSubmit(Request $request)
    {

        $user = auth()->user();

        if ($user->profile_complete == Status::YES) {
            return to_route('user.home');
        }

        $countryData  = (array)json_decode(file_get_contents(resource_path('views/partials/country.json')));
        $countryCodes = implode(',', array_keys($countryData));
        $mobileCodes  = implode(',', array_column($countryData, 'dial_code'));
        $countries    = implode(',', array_column($countryData, 'country'));

        $request->validate([
            'country_code' => 'required|in:' . $countryCodes,
            'country'      => 'required|in:' . $countries,
            'mobile_code'  => 'required|in:' . $mobileCodes,
            'username'     => 'required|unique:users|min:6',
            'mobile'       => ['required', 'regex:/^([0-9]*)$/', Rule::unique('users')->where('dial_code', $request->mobile_code)],
        ]);


        if (preg_match("/[^a-z0-9_]/", trim($request->username))) {
            $notify[] = ['info', 'Username can contain only small letters, numbers and underscore.'];
            $notify[] = ['error', 'No special character, space or capital letters in username.'];
            return back()->withNotify($notify)->withInput($request->all());
        }

        $user->country_code = $request->country_code;
        $user->mobile       = $request->mobile;
        $user->username     = $request->username;


        $user->address = $request->address;
        $user->city = $request->city;
        $user->state = $request->state;
        $user->zip = $request->zip;
        $user->country_name = @$request->country;
        $user->dial_code = $request->mobile_code;

        $user->profile_complete = Status::YES;
        $user->save();

        return to_route('user.home');
    }


    public function addDeviceToken(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'token' => 'required',
        ]);

        if ($validator->fails()) {
            return ['success' => false, 'errors' => $validator->errors()->all()];
        }

        $deviceToken = DeviceToken::where('token', $request->token)->first();

        if ($deviceToken) {
            return ['success' => true, 'message' => 'Already exists'];
        }

        $deviceToken          = new DeviceToken();
        $deviceToken->user_id = auth()->user()->id;
        $deviceToken->token   = $request->token;
        $deviceToken->is_app  = Status::NO;
        $deviceToken->save();

        return ['success' => true, 'message' => 'Token saved successfully'];
    }

    public function downloadAttachment($fileHash)
    {
        $filePath = decrypt($fileHash);
        $extension = pathinfo($filePath, PATHINFO_EXTENSION);
        $title = slug(gs('site_name')) . '- attachments.' . $extension;
        try {
            $mimetype = mime_content_type($filePath);
        } catch (\Exception $e) {
            $notify[] = ['error', 'File does not exists'];
            return back()->withNotify($notify);
        }
        header('Content-Disposition: attachment; filename="' . $title);
        header("Content-Type: " . $mimetype);
        return readfile($filePath);
    }

    public function becomeInstructor()
    {
        $pageTitle = "Become Instructor";
        if (auth()->user()->is_instructor == Status::PENDING) {
            $notify[] = ['error', 'Sorry! You have already applied for instructor'];
            return redirect(route('user.home'))->withNotify($notify);
        }
        if (auth()->user()->is_instructor == Status::APPROVED) {
            $notify[] = ['error', 'Sorry! You are already an instructor'];
            return redirect(route('user.home'))->withNotify($notify);
        }
        return view('Template::user.become_instructor', compact('pageTitle'));
    }

    public function applyAsInstructor(Request $request)
    {
        $request->validate([
            'fname' => 'required',
            'lname' => 'required',
            'email' => 'required|email',
            'mobile' => 'required',
            'occupation' => 'required',
            'detail' => 'required',
            'resume' => ['required', 'max:5120', new FileTypeValidate(['pdf'])],
        ]);

        $user = auth()->user();
        if ($request->resume) {
            $resume = fileUploader($request->resume, getFilePath('resume'));
        }
        $user->instructor_info = [
            'firstname' => $request->fname,
            'lastname' => $request->lname,
            'email' => $request->email,
            'mobile' => $request->mobile,
            'occupation' => $request->occupation,
            'detail' => $request->detail,
            'resume' => $resume
        ];
        $user->is_instructor = Status::PENDING;
        $user->save();

        $adminNotification = new AdminNotification();
        $adminNotification->user_id = $user->id;
        $adminNotification->title = 'New application for instructor.';
        $adminNotification->click_url = urlPath('admin.instructors.pending');
        $adminNotification->save();

        $notify[] = ['success', 'Your Application has been submitted for review'];
        return redirect(route('user.home'))->withNotify($notify);
    }

    public function postComment(Request $request)
    {
        $request->validate([
            'comment' => 'required',
            'course_id' => 'required|numeric|gt:0'
        ]);

        $comment            = new Comment();
        $comment->user_id   = auth()->id();
        $comment->course_id = $request->course_id;
        $comment->parent_id = $request->parent_id ?? 0;
        $comment->comment   = $request->comment;
        $comment->save();

        $notify[] = ['success', 'Comment has been posted'];
        return back()->withNotify($notify);
    }

    public function applyCoupon(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'coupon' => 'required'
        ]);
        if ($validate->fails()) {
            return response()->json($validate->errors());
        }
        $coupon = Coupon::where('coupon_code', '=', strtoupper($request->coupon))->where('status', '=', 1)->first();

        if (!$coupon) {
            return response()->json(['coupon' => ['Sorry! Invalid coupon']]);
        }

        if ($coupon->use_limit <= 0) {
            return response()->json(['coupon' => ['Sorry! Coupon limit has been reached']]);
        }
        if ($coupon->start_date > Carbon::now()->toDateString()) {
            return response()->json(['coupon' => ['Sorry! Coupon is not valid in this date']]);
        }
        if ($coupon->end_date < Carbon::now()->toDateString()) {
            $coupon->status = Status::DISABLE;
            $coupon->update();

            return response()->json(['coupon' => ['Sorry! Coupon has been expired']]);
        }


        $general = gs();
        $course  = Course::find($request->course_id);
        $price   = $course->discount ? $course->discountPrice() : $course->price;
        if (!$course) {
            return response()->json(['coupon' => ['Sorry! Something went wrong']]);
        }
        $couponUser = CouponUser::where('user_id', auth()->id())->where('coupon_id', $coupon->id)->get();

        if ($price < $coupon->min_order_amount) {
            return response()->json(['coupon' => ["Sorry! Minimum course price is required for this coupon is " . getAmount($coupon->min_order_amount) . ' ' . $general->cur_text]]);
        }

        if ($couponUser->count() >= $coupon->usage_per_user) {
            return response()->json(['coupon' => ['Sorry! Your Coupon limit has been reached']]);
        } else {
            $couponUser = new CouponUser();
            $couponUser->user_id = auth()->id();
            $couponUser->coupon_id = $coupon->id;
            $couponUser->save();

            if ($coupon->use_limit > 0) {
                $coupon->use_limit--;
                $coupon->save();
            }
        }

        if ($coupon->course_id == 0) {
            if ($coupon->amount_type == 2) {
                $newPrice = $price - $coupon->coupon_amount;
            } else {
                $discount = $price * ($coupon->coupon_amount / 100);
                $newPrice = $price - $discount;
            }
            $newPrice = $newPrice < 0 ? 0 : $newPrice;
            session()->put('newPrice', $newPrice);
            session()->put('coupon', $coupon->coupon_code);
            return response()->json(
                [
                    'yes' => "Coupon applied! new price is $newPrice" . $general->cur_text,
                    'newPrice' => "$newPrice"
                ]
            );
        } else {

            if ($coupon->course_id != $course->id) {
                return response()->json(['coupon' => ['Sorry! Coupon not valid for this course']]);
            } else {
                if ($coupon->amount_type == 2) {
                    $newPrice = $price - $coupon->coupon_amount;
                } else {
                    $discount = $price * ($coupon->coupon_amount / 100);
                    $newPrice = $price - $discount;
                }
                $newPrice = $newPrice < 0 ? 0 : $newPrice;
                session()->put('newPrice', $newPrice);
                session()->put('coupon', $coupon->coupon_code);
                return response()->json(
                    [
                        'yes' => "Coupon applied! new price is $newPrice" . $general->cur_text,
                        'newPrice' => "$newPrice"
                    ]
                );
            }
        }
    }
}
