<?php

namespace App\Providers;

use App\Models\Page;
use App\Models\User;
use App\Lib\Searchable;
use App\Models\Deposit;
use App\Models\Category;
use App\Models\Frontend;
use App\Models\Language;
use App\Constants\Status;
use App\Models\Withdrawal;
use App\Models\SupportTicket;
use App\Models\AdminNotification;
use App\Models\Course;
use Illuminate\Pagination\Paginator;
use Illuminate\Support\ServiceProvider;
use Illuminate\Database\Eloquent\Builder;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
        Builder::mixin(new Searchable);
    }

    /**
     * Bootstrap any application services.
     */
    public function boot(): void
    {
        if (!cache()->get('SystemInstalled'))
        {
            $envFilePath = base_path('.env');
            if (!file_exists($envFilePath))
            {
                header('Location: install');
                exit;
            }
            $envContents = file_get_contents($envFilePath);
            if (empty($envContents))
            {
                header('Location: install');
                exit;
            }
            else
            {
                cache()->put('SystemInstalled', true);
            }
        }


        $activeTemplate = activeTemplate();
        $viewShare['activeTemplate'] = $activeTemplate;
        $viewShare['activeTemplateTrue'] = activeTemplate(true);
        $viewShare['emptyMessage'] = 'Data not found';

        $viewShare['language'] = Language::all();
        $viewShare['pages'] = Page::where('tempname', $activeTemplate)->where('is_default', Status::NO)->get();

        view()->share($viewShare);


        view()->composer('admin.partials.sidenav', function ($view)
        {
            $view->with([
                'bannedUsersCount'                 => User::banned()->count(),
                'emailUnverifiedUsersCount'        => User::emailUnverified()->count(),
                'mobileUnverifiedUsersCount'       => User::mobileUnverified()->count(),
                'pendingTicketCount'               => SupportTicket::whereIN('status', [Status::TICKET_OPEN, Status::TICKET_REPLY])->count(),
                'pendingDepositsCount'             => Deposit::pending()->count(),
                'pendingWithdrawCount'             => Withdrawal::pending()->count(),
                'updateAvailable'                  => version_compare(gs('available_version'), systemDetails()['version'], '>') ? 'v' . gs('available_version') : false,
                'bannedInstructorsCount'           => User::where('is_instructor', Status::APPROVED)->banned()->count(),
                'emailUnverifiedInstructorsCount'  => User::where('is_instructor', Status::APPROVED)->emailUnverified()->count(),
                'mobileUnverifiedInstructorsCount' => User::where('is_instructor', Status::APPROVED)->mobileUnverified()->count(),
                'kycUnverifiedInstructorsCount'    => User::where('is_instructor', Status::APPROVED)->kycUnverified()->count(),
                'kycPendingInstructorsCount'       => User::where('is_instructor', Status::APPROVED)->kycPending()->count(),
                'pendingInstructorsCount'          => User::where('is_instructor', Status::PENDING)->count(),
                'pendingCoursesCount'              => Course::where('status', Status::INACTIVE)->count(),
            ]);
        });

        view()->composer('admin.partials.topnav', function ($view)
        {
            $view->with([
                'adminNotifications' => AdminNotification::where('is_read', Status::NO)->with('user')->orderBy('id', 'desc')->take(10)->get(),
                'adminNotificationCount' => AdminNotification::where('is_read', Status::NO)->count(),
            ]);
        });

        view()->composer('partials.seo', function ($view)
        {
            $seo = Frontend::where('data_keys', 'seo.data')->first();
            $view->with([
                'seo' => $seo ? $seo->data_values : $seo,
            ]);
        });

        view()->composer([$activeTemplate . 'partials.header'], function ($view)
        {
            $view->with([
                'categories' => Category::where('status', Status::ACTIVE)->whereHas('courses')
                    ->with(['subcategories' => function ($query)
                    {
                        $query->whereHas('courses');
                    }])
                    ->latest()->get(['name', 'id']),
            ]);
        });

        if (gs('force_ssl'))
        {
            \URL::forceScheme('https');
        }


        Paginator::useBootstrapFive();
    }
}
